package com.stylusstudio.msv;

import java.io.IOException;
import java.util.ResourceBundle;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import com.sun.msv.driver.textui.DebugController;
import com.sun.msv.driver.textui.ReportErrorHandler;
import com.sun.msv.grammar.Grammar;
import com.sun.msv.grammar.xmlschema.XMLSchemaGrammar;
import com.sun.msv.reader.dtd.DTDReader;
import com.sun.msv.reader.util.GrammarLoader;
import com.sun.msv.reader.util.IgnoreController;
import com.sun.msv.relaxns.grammar.RELAXGrammar;
import com.sun.msv.relaxns.verifier.SchemaProviderImpl;
import com.sun.msv.util.Util;
import com.sun.msv.verifier.ValidationUnrecoverableException;
import com.sun.msv.verifier.regexp.REDocumentDeclaration;

/*
 * Copyright (c) 2006 Progress Software Corporation. All rights reserved.
 * http://www.progress.com/
 *
 * Stylus Studio
 * http://www.stylusstudio.com/
 */

public class StylusDriver {

	public StylusDriver(String source) {
		m_source = source;
	}

	public static void main(String[] args) {

	    System.out.print("Stylus Studio Driver for the Sun MSV Validator ");
	    System.out.println(ResourceBundle.getBundle("version").getString("version"));

		for (int i = 0; i < args.length; i++) {
			if (i > 0)
				System.out.println("------------------------------------------------------------------------");
			System.out.println("--Starting validation of \"" + args[i] + "\"--");
			new StylusDriver(args[i]).run();
			System.out.println("--Completed validation of \"" + args[i] + "\"--");
		}
	}


	private String m_source;
	
	public boolean run() {
		XMLReader reader;
		try {
			reader = XMLReaderFactory.createXMLReader();
		} catch (SAXException saxe) {
			saxe.printStackTrace(System.err);
			return false;
		}
		DialectHandler dh = new DialectHandler();
		reader.setContentHandler(dh);
		try {
			reader.setProperty("http://xml.org/sax/properties/lexical-handler", dh);
		} catch (SAXNotRecognizedException saxnre) {
			saxnre.printStackTrace(System.err);
			return false;
		} catch (SAXNotSupportedException saxnse) {
			saxnse.printStackTrace(System.err);
			return false;
		}

		try {
			reader.parse(m_source);
		} catch (IOException ioe) {
			ioe.printStackTrace(System.err);
			return false;
		} catch (FoundNeededInfoException fnie) {
			
		} catch (SAXException saxe) {
			System.out.print(saxe.getMessage());
			return true;
		}

		String dtd = dh.getSystemId();
		String xsd = dh.getXsdUri();

		if (dtd == null && xsd == null) {
			System.out.println("No identifiable DTD or XML Schema in " + m_source);
			return true;
		}

		// get a new reader for actual validation
		SAXParserFactory factory = SAXParserFactory.newInstance();
		factory.setNamespaceAware(true);
		factory.setValidating(false);

		if (dtd != null) {
			System.out.println("Validating \"" + m_source + "\" against DTD \"" + dtd + '"');
			if (validate(factory, m_source, dtd, true))
				return true;
		}

		if (dh.getXsdUri() != null) {
			try {
				factory.setFeature("http://apache.org/xml/features/validation/dynamic",true);
				// turn off XML Schema validation if Xerces is used
				factory.setFeature("http://apache.org/xml/features/validation/schema",false);
			} catch (SAXNotRecognizedException saxnre) {
				// ignore
			} catch (SAXNotSupportedException saxnse) {
				// ignore
			} catch (ParserConfigurationException pce) {
				// ignore
			}
			System.out.println("Validating \"" + m_source + "\" against XML Schema \"" + xsd + '"');
			if (validate(factory, m_source, xsd, false))
				return true;
		}

		return false; // no non-schema-related errors
	}


	private boolean validate(SAXParserFactory factory, String source, String schema, boolean isDTD)
	{
		boolean warning = false;
		boolean strict  = false;
		Grammar grammar = null;
		try {
			if (isDTD) {
				grammar = DTDReader.parse(new InputSource(schema), new IgnoreController());
			} else {
				GrammarLoader loader = new GrammarLoader();
				// set various parameters
				loader.setController(new DebugController(warning, false));
				loader.setSAXParserFactory(factory);
				loader.setStrictCheck(strict);
				grammar = loader.parse(schema);
			}			
		} catch (SAXParseException saxpe) {
			System.out.println(saxpe.getMessage());
			return true;
		} catch (SAXException saxe) {
			System.out.println(saxe.getMessage());
			return true;
		} catch (ParserConfigurationException pce) {
			pce.printStackTrace(System.err);
			return true;
		} catch (IOException ioe) {
			ioe.printStackTrace(System.err);
			return true;
		}
		
		if (grammar==null) {
			System.out.println("Grammar \"" + schema + "\" could not be loaded");
			return false;
		}
			
		DocumentVerifier verifier;
		if (grammar instanceof RELAXGrammar)
			// use divide&validate framework to validate document
			verifier = new RelaxVerifier(new SchemaProviderImpl((RELAXGrammar)grammar));
		else
		if (grammar instanceof XMLSchemaGrammar )
			// use verifier+identity constraint checker.
			verifier = new XMLSchemaVerifier((XMLSchemaGrammar)grammar);
		else
			// validate normally by using Verifier.
			verifier = new SimpleVerifier(new REDocumentDeclaration(grammar));
		
		boolean result = false;
		
		try {
			XMLReader reader = factory.newSAXParser().getXMLReader();
			reader.setErrorHandler(new ReportErrorHandler());
			result = verifier.verify(reader, Util.getInputSource(m_source), false);
		} catch (ValidationUnrecoverableException vue) {
			System.out.println("Unrecoverable validation error. Bailing from validator.");
		} catch (SAXParseException saxpe) {
            if (saxpe.getException() != null)
            	saxpe.getException().printStackTrace(System.err);
			; // error is already reported by ErrorHandler
		} catch (SAXException saxe) {
            if (saxe.getException() != null)
            	saxe.getException().printStackTrace(System.err);
		} catch (ParserConfigurationException pce) {
			pce.printStackTrace(System.err);
			return true;
		} catch (Exception e) {
			e.printStackTrace(System.err);
			return true;
		}

		if (result)
			System.out.println('"' + m_source + "\" is valid");
        else
			System.out.println('"' + m_source + "\" is NOT valid");

		return false; // no non-schema-related errors
	}
}
